<?php
require_once dirname(__DIR__, 2) . 
'/config/config.php';

// Check if the user is logged in
if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("location: login.php");
    exit;
}

$user_id = $_SESSION["id"];
$stream_key = "";
$stream_title = "";
$stream_description = "";
$message = "";

// Check if user already has stream settings
$sql_get = "SELECT stream_key, title, description FROM streams WHERE user_id = :user_id";
if ($stmt_get = $pdo->prepare($sql_get)) {
    $stmt_get->bindParam(":user_id", $user_id, PDO::PARAM_INT);
    if ($stmt_get->execute()) {
        $stream_data = $stmt_get->fetch();
        if ($stream_data) {
            $stream_key = $stream_data["stream_key"];
            $stream_title = $stream_data["title"];
            $stream_description = $stream_data["description"];
        }
    }
    unset($stmt_get);
}

// Handle form submission for updating settings or generating key
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $new_title = trim($_POST["title"]);
    $new_description = trim($_POST["description"]);
    $regenerate_key = isset($_POST["regenerate_key"]);

    // Generate a new key if requested or if none exists
    if (empty($stream_key) || $regenerate_key) {
        // Generate a reasonably unique key (simple example)
        $new_stream_key = bin2hex(random_bytes(16)); 
    } else {
        $new_stream_key = $stream_key; // Keep existing key
    }

    if (empty($new_title)) {
        $new_title = $_SESSION["username"] . "'s Stream"; // Default title
    }

    // Upsert logic: Update if exists, Insert if not
    $sql_upsert = "INSERT INTO streams (user_id, stream_key, title, description, is_live, created_at) 
                   VALUES (:user_id, :stream_key, :title, :description, 0, datetime('now'))
                   ON CONFLICT(user_id) DO UPDATE SET 
                   stream_key = excluded.stream_key, 
                   title = excluded.title, 
                   description = excluded.description";
                   
    if ($stmt_upsert = $pdo->prepare($sql_upsert)) {
        $stmt_upsert->bindParam(":user_id", $user_id, PDO::PARAM_INT);
        $stmt_upsert->bindParam(":stream_key", $new_stream_key, PDO::PARAM_STR);
        $stmt_upsert->bindParam(":title", $new_title, PDO::PARAM_STR);
        $stmt_upsert->bindParam(":description", $new_description, PDO::PARAM_STR);

        if ($stmt_upsert->execute()) {
            $message = "Stream settings updated successfully.";
            // Update local variables for display
            $stream_key = $new_stream_key;
            $stream_title = $new_title;
            $stream_description = $new_description;
            if ($regenerate_key) {
                $message .= " New stream key generated.";
            }
        } else {
            $message = "Error updating stream settings.";
        }
        unset($stmt_upsert);
    }
}

unset($pdo);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Stream Settings - Cortlak.com</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body { font-family: sans-serif; background-color: #f0f0f0; margin: 0; padding: 20px; }
        .container { max-width: 600px; margin: 20px auto; background-color: #fff; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h2 { text-align: center; color: #333; margin-bottom: 20px; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; color: #555; }
        input[type="text"], textarea { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; font-family: monospace; }
        textarea { resize: vertical; min-height: 80px; font-family: sans-serif; }
        .help-text { font-size: 0.9em; color: #666; margin-top: 5px; }
        .message { padding: 10px; margin-bottom: 15px; border-radius: 4px; }
        .message.success { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .message.error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .btn { display: inline-block; padding: 10px 20px; background-color: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 1em; text-align: center; text-decoration: none; }
        .btn:hover { background-color: #0056b3; }
        .btn-danger { background-color: #dc3545; }
        .btn-danger:hover { background-color: #c82333; }
        .key-display { background-color: #e9ecef; padding: 10px; border-radius: 4px; margin-bottom: 10px; word-break: break-all; }
        .flex-container { display: flex; justify-content: space-between; align-items: center; }
    </style>
</head>
<body>

<div class="container">
    <h2>Stream Settings</h2>

    <?php if (!empty($message)): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post">
        <div class="form-group">
            <label for="title">Stream Title</label>
            <input type="text" id="title" name="title" value="<?php echo htmlspecialchars($stream_title); ?>" placeholder="My Awesome Stream">
        </div>

        <div class="form-group">
            <label for="description">Stream Description</label>
            <textarea id="description" name="description" placeholder="Playing some games and chatting!"><?php echo htmlspecialchars($stream_description); ?></textarea>
        </div>

        <div class="form-group">
            <label>Stream Key</label>
            <?php if (!empty($stream_key)): ?>
                <div class="key-display"><code><?php echo htmlspecialchars($stream_key); ?></code></div>
                <p class="help-text">Keep this key secret! Use it in your streaming software (e.g., OBS). If you suspect it's compromised, regenerate it.</p>
                <button type="submit" name="regenerate_key" value="1" class="btn btn-danger" onclick="return confirm(
'Are you sure you want to generate a new stream key? Your old key will stop working immediately.
');">Regenerate Key</button>
            <?php else: ?>
                <p class="help-text">You don't have a stream key yet. Save your settings to generate one.</p>
            <?php endif; ?>
        </div>

        <div class="form-group flex-container">
            <input type="submit" class="btn" value="Save Settings">
            <a href="profile.php" style="color: #6c757d;">Back to Profile</a>
        </div>
    </form>

    <hr style="margin: 20px 0;">

    <h3>Streaming Setup (Example for OBS)</h3>
    <p>To stream to Cortlak.com, configure your streaming software (like OBS Studio) with the following settings:</p>
    <ul>
        <li><strong>Service:</strong> Custom...</li>
        <li><strong>Server:</strong> rtmp://your-rtmp-server-address/live (Replace with actual RTMP server address when implemented)</li>
        <li><strong>Stream Key:</strong> <code><?php echo htmlspecialchars($stream_key ?: '[Your Stream Key]'); ?></code> (Copy the key from above)</li>
    </ul>
    <p><small>Note: An RTMP server needs to be set up separately to handle the actual video stream ingest and distribution. This example only provides the key management part.</small></p>

</div>

</body>
</html>
